#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest

from .. import server
from ..generic import to_string
from .mock_helper import ModulePatches


class TestPhonehome(unittest.TestCase):

    demo_status_old = (
        'Phone-home active. Checking PID.'
        '\nServer ext1.perfact.de: 2959'
        '\nServer ext2.perfact.de: -'
    )

    demo_status_new = (
        '● perfact-phonehome.service - Phone Home, connection to PerFact '
        'gateway, necessary for PerFact Support'
        '\n     Loaded: loaded '
        '(/lib/systemd/system/perfact-phonehome.service; '
        'enabled; vendor preset: enabled)'
        '\n     Active: activating (auto-restart) (Result: exit-code) since '
        'Mon 2021-08-02 14:10:57 CEST; 1s ago'
        '\n    Process: 3990 ExecStart=/usr/bin/perfact-phonehome (code=exited'
        ', status=255/EXCEPTION)'
        '\n   Main PID: 3990 (code=exited, status=255/EXCEPTION)'
    )

    # patch socket's recv method to return a dummy value
    @staticmethod
    def patchSyscall(mobj, version, output):
        syscall = mobj.safe_syscall

        def mock_syscall(commands):
            # version check
            if (commands[0].startswith('sh') and len(commands) > 2
                    and commands[2].startswith('dpkg')):
                return (0, version)
            # normal command
            return (0, output)

        syscall.side_effect = mock_syscall
        return syscall

    # test for various versions
    def test_phonehome_versions(self):
        with ModulePatches(server, ['safe_syscall']) as mobj:

            # Version 1.0
            syscall = self.patchSyscall(mobj, '1.0', self.demo_status_old)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with(['/usr/bin/phonehome', 'status'])

            # Version 1.16.4
            syscall = self.patchSyscall(mobj, '1.16.4', self.demo_status_old)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with(['/usr/bin/phonehome', 'status'])

            # Version 2.0.0
            syscall = self.patchSyscall(mobj, '2.0.0', self.demo_status_new)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with([
                'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
            ])

            # Version 3.0.0
            syscall = self.patchSyscall(mobj, '2.0.0', self.demo_status_new)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with([
                'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
            ])

    # test the status command
    def test_phonehome_status(self):
        with ModulePatches(server, ['safe_syscall']) as mobj:
            # Test old version
            syscall = self.patchSyscall(mobj, '1.16.4', self.demo_status_old)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with(['/usr/bin/phonehome', 'status'])
            # assert output
            assert output == to_string(self.demo_status_old)

            # Test new version
            syscall = self.patchSyscall(mobj, '2.0.0', self.demo_status_new)
            retcode, output = server.phonehome_status()
            # assert call
            syscall.assert_called_with([
                'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
            ])
            # assert output
            assert output == to_string(self.demo_status_new)

    # test the up command
    def test_phonehome_up(self):
        with ModulePatches(server, ['safe_syscall']) as mobj:
            # Test old version
            syscall = self.patchSyscall(mobj, '1.16.4', '')
            retcode, output = server.phonehome_up()
            # assert call
            syscall.assert_called_with(['/usr/bin/phonehome', 'status'])

            # Test new version
            syscall = self.patchSyscall(mobj, '2.0.0', '')
            retcode, output = server.phonehome_up()
            # assert call
            syscall.assert_called_with([
                'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
            ])

    # test the down command
    def test_phonehome_down(self):
        with ModulePatches(server, ['safe_syscall']) as mobj:
            # Test old version
            syscall = self.patchSyscall(mobj, '1.16.4', '')
            retcode, output = server.phonehome_down()
            # assert call
            syscall.assert_called_with(['/usr/bin/phonehome', 'status'])

            # Test new version
            syscall = self.patchSyscall(mobj, '2.0.0', '')
            retcode, output = server.phonehome_down()
            # assert call
            syscall.assert_called_with([
                'sudo', 'systemctl', 'status', 'perfact-phonehome.service'
            ])
